//	Altirra - Atari 800/800XL emulator
//	Copyright (C) 2008 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "stdafx.h"
#include "simulator.h"
#include "console.h"
#include "debugger.h"
#include "symbols.h"

extern ATSimulator g_sim;

namespace {
	enum {
		kModeInvalid,
		kModeImplied,
		kModeRel,
		kModeImm,
		kModeImm16,
		kModeZp,
		kModeZpX,
		kModeZpY,
		kModeAbs,
		kModeAbsX,
		kModeAbsY,
		kModeIndA,
		kModeIndX,
		kModeIndY
	};

	static const uint8 kBytesPerMode[]={
		1,1,2,2,3,2,2,2,3,3,3,3,2,2
	};

	enum {
		kOpcodebad,
		kOpcodeADC,
		kOpcodeANC,
		kOpcodeAND,
		kOpcodeASL,
		kOpcodeASR,
		kOpcodeBCC,
		kOpcodeBCS,
		kOpcodeBEQ,
		kOpcodeBIT,
		kOpcodeBMI,
		kOpcodeBNE,
		kOpcodeBPL,
		kOpcodeBVC,
		kOpcodeBVS,
		kOpcodeCLC,
		kOpcodeCLD,
		kOpcodeCLI,
		kOpcodeCLV,
		kOpcodeCMP,
		kOpcodeCPX,
		kOpcodeCPY,
		kOpcodeDCP,
		kOpcodeDEC,
		kOpcodeDEX,
		kOpcodeDEY,
		kOpcodeEOR,
		kOpcodeHLE,
		kOpcodeINC,
		kOpcodeINX,
		kOpcodeINY,
		kOpcodeISB,
		kOpcodeJMP,
		kOpcodeJSR,
		kOpcodeLAX,
		kOpcodeLDA,
		kOpcodeLDX,
		kOpcodeLDY,
		kOpcodeLSR,
		kOpcodeNOP,
		kOpcodeORA,
		kOpcodePHA,
		kOpcodePHP,
		kOpcodePLA,
		kOpcodePLP,
		kOpcodeRLA,
		kOpcodeROL,
		kOpcodeROR,
		kOpcodeRTI,
		kOpcodeRTS,
		kOpcodeSAX,
		kOpcodeSBC,
		kOpcodeSEC,
		kOpcodeSED,
		kOpcodeSEI,
		kOpcodeSLO,
		kOpcodeSRE,
		kOpcodeSTA,
		kOpcodeSTX,
		kOpcodeSTY,
		kOpcodeTAX,
		kOpcodeTAY,
		kOpcodeTSX,
		kOpcodeTXA,
		kOpcodeTXS,
		kOpcodeTYA
	};

	static const char *kOpcodes[]={
		"bad",
		"ADC",
		"ANC",
		"AND",
		"ASL",
		"ASR",
		"BCC",
		"BCS",
		"BEQ",
		"BIT",
		"BMI",
		"BNE",
		"BPL",
		"BVC",
		"BVS",
		"CLC",
		"CLD",
		"CLI",
		"CLV",
		"CMP",
		"CPX",
		"CPY",
		"DCP",
		"DEC",
		"DEX",
		"DEY",
		"EOR",
		"HLE",
		"INC",
		"INX",
		"INY",
		"ISB",
		"JMP",
		"JSR",
		"LAX",
		"LDA",
		"LDX",
		"LDY",
		"LSR",
		"NOP",
		"ORA",
		"PHA",
		"PHP",
		"PLA",
		"PLP",
		"RLA",
		"ROL",
		"ROR",
		"RTI",
		"RTS",
		"SAX",
		"SBC",
		"SEC",
		"SED",
		"SEI",
		"SLO",
		"SRE",
		"STA",
		"STX",
		"STY",
		"TAX",
		"TAY",
		"TSX",
		"TXA",
		"TXS",
		"TYA",
	};

	#define xx(op) { kModeInvalid, 0 }
	#define Ip(op) { kModeImplied, kOpcode##op }
	#define Re(op) { kModeRel, kOpcode##op }
	#define Im(op) { kModeImm, kOpcode##op }
	#define I2(op) { kModeImm16, kOpcode##op }
	#define Zp(op) { kModeZp, kOpcode##op }
	#define Zx(op) { kModeZpX, kOpcode##op }
	#define Zy(op) { kModeZpY, kOpcode##op }
	#define Ab(op) { kModeAbs, kOpcode##op }
	#define Ax(op) { kModeAbsX, kOpcode##op }
	#define Ay(op) { kModeAbsY, kOpcode##op }
	#define Ia(op) { kModeIndA, kOpcode##op }
	#define Ix(op) { kModeIndX, kOpcode##op }
	#define Iy(op) { kModeIndY, kOpcode##op }

	uint8 kModeTbl[256][2]={
		//			   0,       1,       2,       3,       4,       5,       6,       7,       8,       9,       A,       B,       C,       D,       E,       F
		/* 00 */	xx(bad), Ix(ORA), I2(HLE), Ix(SLO), Zp(NOP), Zp(ORA), Zp(ASL), Zp(SLO), Ip(PHP), Im(ORA), Ip(ASL), Im(ANC), Ab(NOP), Ab(ORA), Ab(ASL), Ab(SLO), 
		/* 10 */	Re(BPL), Iy(ORA), xx(bad), xx(bad), Zx(NOP), Zx(ORA), Zx(ASL), Zx(SLO), Ip(CLC), Ay(ORA), Ip(NOP), xx(bad), Ax(NOP), Ax(ORA), Ax(ASL), Ax(SLO), 
		/* 20 */	Ab(JSR), Ix(AND), xx(bad), xx(bad), Zp(BIT), Zp(AND), Zp(ROL), xx(bad), Ip(PLP), Im(AND), Ip(ROL), Im(ANC), Ab(BIT), Ab(AND), Ab(ROL), xx(bad), 
		/* 30 */	Re(BMI), Iy(AND), xx(bad), xx(bad), xx(bad), Zx(AND), Zx(ROL), Zx(RLA), Ip(SEC), Ay(AND), Ip(NOP), xx(bad), Ax(NOP), Ax(AND), Ax(ROL), xx(bad), 
		/* 40 */	Ip(RTI), Ix(EOR), xx(bad), Ix(SRE), Zp(NOP), Zp(EOR), Zp(LSR), Zp(SRE), Ip(PHA), Im(EOR), Ip(LSR), Im(ASR), Ab(JMP), Ab(EOR), Ab(LSR), Ab(SRE), 
		/* 50 */	Re(BVC), Iy(EOR), xx(bad), Iy(SRE), Zx(NOP), Zx(EOR), Zx(LSR), Zx(SRE), Ip(CLI), Ay(EOR), Ip(NOP), Ay(SRE), Ax(NOP), Ax(EOR), Ax(LSR), Ax(SRE), 
		/* 60 */	Ip(RTS), Ix(ADC), xx(bad), xx(bad), xx(bad), Zp(ADC), Zp(ROR), xx(bad), Ip(PLA), Im(ADC), Ip(ROR), xx(bad), Ia(JMP), Ab(ADC), Ab(ROR), xx(bad), 
		/* 70 */	Re(BVS), Iy(ADC), xx(bad), xx(bad), xx(bad), Zx(ADC), Zx(ROR), xx(bad), Ip(SEI), Ay(ADC), Ip(NOP), xx(bad), Ax(NOP), Ax(ADC), Ax(ROR), xx(bad), 
		/* 80 */	Im(NOP), Ix(STA), xx(bad), xx(bad), Zp(STY), Zp(STA), Zp(STX), xx(bad), Ip(DEY), Im(STA), Ip(TXA), xx(bad), Ab(STY), Ab(STA), Ab(STX), Ab(SAX), 
		/* 90 */	Re(BCC), Iy(STA), xx(bad), xx(bad), Zx(STY), Zx(STA), Zy(STX), xx(bad), Ip(TYA), Ay(STA), Ip(TXS), xx(bad), xx(bad), Ax(STA), Ay(STX), xx(bad), 
		/* A0 */	Im(LDY), Ix(LDA), Im(LDX), xx(bad), Zp(LDY), Zp(LDA), Zp(LDX), xx(bad), Ip(TAY), Im(LDA), Ip(TAX), xx(bad), Ab(LDY), Ab(LDA), Ab(LDX), Ab(LAX), 
		/* B0 */	Re(BCS), Iy(LDA), xx(bad), xx(bad), Zx(LDY), Zx(LDA), Zy(LDX), xx(bad), Ip(CLV), Ay(LDA), Ip(TSX), xx(bad), Ax(LDY), Ax(LDA), Ay(LDX), Ax(LAX), 
		/* C0 */	Im(CPY), Ix(CMP), xx(bad), Ix(DCP), Zp(CPY), Zp(CMP), Zp(DEC), Zp(DCP), Ip(INY), Im(CMP), Ip(DEX), xx(bad), Ab(CPY), Ab(CMP), Ab(DEC), Ab(DCP), 
		/* D0 */	Re(BNE), Iy(CMP), xx(bad), Iy(DCP), xx(bad), Zx(CMP), Zx(DEC), Zx(DCP), Ip(CLD), Ay(CMP), Ip(NOP), Ay(DCP), Ax(NOP), Ax(CMP), Ax(DEC), Ax(DCP), 
		/* E0 */	Im(CPX), Ix(SBC), xx(bad), Ix(ISB), Zp(CPX), Zp(SBC), Zp(INC), Zp(ISB), Ip(INX), Im(SBC), Ip(NOP), xx(bad), Ab(CPX), Ab(SBC), Ab(INC), Ab(ISB), 
		/* F0 */	Re(BEQ), Iy(SBC), xx(bad), Iy(ISB), Zx(NOP), Zx(SBC), Zx(INC), Zx(ISB), Ip(SED), Ay(SBC), Ip(NOP), Ay(ISB), Ax(NOP), Ax(SBC), Ax(INC), Ax(ISB),
	};
}

const char *ATGetSymbolName(uint16 addr, bool write) {
	IATDebuggerSymbolLookup *symlookup = ATGetDebuggerSymbolLookup();

	ATSymbol sym;
	if (!symlookup->LookupSymbol(addr, write ? kATSymbol_Write : kATSymbol_Read | kATSymbol_Execute, sym))
		return NULL;

	return sym.mpName;
}

uint16 ATDisassembleInsn(char *buf, uint16 addr) {
	uint8 opcode = g_sim.DebugReadByte(addr);
	uint8 byte1 = g_sim.DebugReadByte(addr+1);
	uint8 byte2 = g_sim.DebugReadByte(addr+2);

	uint8 mode = kModeTbl[opcode][0];
	uint8 opid = kModeTbl[opcode][1];
	const char *opname = kOpcodes[opid];

	ATCPUEmulator& cpu = g_sim.GetCPU();

	VDStringA line;

	line.sprintf("%04X:", addr);

	int opsize = kBytesPerMode[mode];
	switch(opsize) {
		case 1:
			line.append_sprintf(" %02X      ", opcode);
			break;
		case 2:
			line.append_sprintf(" %02X %02X   ", opcode, byte1);
			break;
		case 3:
			line.append_sprintf(" %02X %02X %02X", opcode, byte1, byte2);
			break;
	}

	const char *label = ATGetSymbolName(addr, false);
	line.append_sprintf("  %-6s %s", label ? label : "", opname);

	if (mode == kModeImm) {
		line.append_sprintf(" #$%02X", byte1);
	} else if (mode == kModeImm16) {
		line.append_sprintf(" #$%02X%02X", byte2, byte1);
	} else if (mode != kModeInvalid && mode != kModeImplied) {
		line += ' ';

		switch(mode) {
			case kModeIndA:
			case kModeIndX:
			case kModeIndY:
				line += '(';
				break;
		}

		uint16 base;
		uint16 ea;
		bool addr16 = false;
		bool ea16 = false;

		switch(mode) {
			case kModeRel:
				base = ea = addr + 2 + (sint8)byte1;
				addr16 = true;
				ea16 = true;
				break;

			case kModeZp:
				base = ea = byte1;
				break;

			case kModeZpX:
				base = byte1;
				ea = (uint8)(byte1+cpu.GetX());
				break;

			case kModeZpY:
				base = byte1;
				ea = (uint8)(byte1+cpu.GetY());
				break;

			case kModeAbs:
				base = ea = byte1 + (byte2 << 8);
				addr16 = true;
				ea16 = true;
				break;

			case kModeAbsX:
				base = byte1 + (byte2 << 8);
				ea = base + cpu.GetX();
				addr16 = true;
				ea16 = true;
				break;

			case kModeAbsY:
				base = byte1 + (byte2 << 8);
				ea = base + cpu.GetY();
				addr16 = true;
				ea16 = true;
				break;

			case kModeIndA:
				base = byte1 + (byte2 << 8);
				ea = g_sim.DebugReadByte(base) + 256*g_sim.DebugReadByte(base+1);
				addr16 = true;
				ea16 = true;
				break;

			case kModeIndX:
				base = byte1;
				ea = g_sim.DebugReadByte((uint8)(base + cpu.GetX())) + 256*g_sim.DebugReadByte((uint8)(base + cpu.GetX() + 1));
				ea16 = true;
				break;

			case kModeIndY:
				base = byte1;
				ea = g_sim.DebugReadByte(base) + 256*g_sim.DebugReadByte((base+1) & 0xff) + cpu.GetY();
				ea16 = true;
				break;
		}

		bool write = false;
		switch(opid) {
		case kOpcodeSTA:
		case kOpcodeSTX:
		case kOpcodeSTY:
			write = true;
			break;
		}

		const char *name = ATGetSymbolName(base, write);
		if (name)
			line.append(name);
		else if (addr16)
			line.append_sprintf("$%04X", base);
		else
			line.append_sprintf("$%02X", base);

		switch(mode) {
			case kModeZpX:
			case kModeAbsX:
				line.append(",X");
				break;

			case kModeZpY:
			case kModeAbsY:
				line.append(",Y");
				break;

			case kModeIndA:
				line.append(")");
				break;

			case kModeIndX:
				line.append(",X)");
				break;

			case kModeIndY:
				line.append("),Y");
				break;
		}

		if (mode != kModeRel) {
			if (line.size() < 30)
				line.resize(30, ' ');

			if (ea16)
				line.append_sprintf(" [$%04X] = $%02X", ea, g_sim.DebugReadByte(ea));
			else
				line.append_sprintf(" [$%02X] = $%02X", ea, g_sim.DebugReadByte(ea));
		}
	}

	line += '\n';
	line += (char)0;
	line.copy(buf, VDStringA::npos);

	return addr + opsize;
}

uint16 ATDisassembleInsn(uint16 addr) {
	char buf[256];
	addr = ATDisassembleInsn(buf, addr);
	ATConsoleWrite(buf);
	return addr;
}

void ATDisassembleRange(FILE *f, uint16 addr1, uint16 addr2) {
	char buf[256];
	while(addr1 < addr2) {
		addr1 = ATDisassembleInsn(buf, addr1);
		fputs(buf, f);
	}
}

uint16 ATDisassembleGetFirstAnchor(uint16 addr, uint16 target) {
	vdfastvector<uint8> results;

	uint16 testbase = addr;
	for(;;) {
		uint16 ip = testbase;
		for(;;) {
			if (ip == target)
				return testbase;

			uint32 offset = (uint16)(ip - addr);
			if (offset < results.size() && results[offset])
				break;

			uint8 opcode = g_sim.DebugReadByte(ip);
			uint8 mode = kModeTbl[opcode][0];

			uint8 oplen = kBytesPerMode[mode];
			if (mode == kModeInvalid || (uint16)(target - ip) < oplen) {
				if (offset >= results.size())
					results.resize(offset+1, false);
				results[offset] = true;
				break;
			}

			ip += oplen;
		}

		++testbase;
		if (testbase == target)
			break;
	}

	return testbase;
}

int ATGetOpcodeLength(uint8 opcode) {
	return kBytesPerMode[kModeTbl[opcode][0]];
}
